<?php
/**
 * Plugin Name: Speedx - Maintenance Mode
 * Description: Maintenance Mode - This minimalist plugin allows administrators to activate maintenance mode with fully customizable HTML and CSS.
 * Version: 9.1
 * Author: Vaclav Pospisil
 * Author URI: https://speedx.cz
 * Requires at least: 6.8
 * Requires PHP: 7.4
 * Updated: 2025-05-19 19:01:00
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
*/

if (!defined('ABSPATH')) {
    exit; // Zabranuje primemu pristupu
}

// Testovaci cookie kdyz nic tak konec
add_action('init', 'spx1mm_set_test_cookie');
function spx1mm_set_test_cookie() {
    if (!headers_sent()) {
        setcookie('spx1mm_test', '1', time() + 3600, "/");
    }
}

// Inicializace prazdne globalni chyby pred dalsim zpracovanim.
if (!isset($GLOBALS['spx1mm_chyba'])) {
    $GLOBALS['spx1mm_chyba'] = '';
}

// Nacteni prekladu
// Until the plugin is approved and the translations are loaded automatically from the Wordpress repository.
function spx1mm_load_textdomain() {
    load_plugin_textdomain('speedx-maintenance-mode', false, dirname(plugin_basename(__FILE__)) . '/languages');
}
add_action('init', 'spx1mm_load_textdomain');

// Nacteni inline javasriptu pro togle buttony
function spx1mm_enqueue_admin_scripts($hook) {
    // Nacti skript jen na strance nastaveni pluginu
    if ($hook !== 'settings_page_spx1mm-settings') {
        return;
    }

    // Registrovat hlavni skript (prazdny, jen jako "handle") 
    wp_register_script('spx1mm-admin-script', '', [], '9', true); // 9 = verze pluginu

    // Inline skripty
    $inline_script = '
        document.addEventListener("DOMContentLoaded", function() {
            // Prepinac Maintenance Mode
            const toggleMM = document.getElementById("spx1mm_enabled");
            const labelMM = document.getElementById("spx1mm_label");
            if (toggleMM && labelMM) {
                function updateLabelMM() {
                    labelMM.textContent = toggleMM.checked ? "' . esc_html__('Disable Maintenance Mode', 'speedx-maintenance-mode') . '" : "' . esc_html__('Enable Maintenance Mode', 'speedx-maintenance-mode') . '";
                }
                toggleMM.addEventListener("change", updateLabelMM);
                updateLabelMM();
            }

            // Prepinac Password Protection
            const togglePW = document.getElementById("spx1mm_enabled_password");
            const labelPW = document.getElementById("spx1mm_label_password");
            if (togglePW && labelPW) {
                function updateLabelPW() {
                    labelPW.textContent = togglePW.checked ? "' . esc_html__('Disable Password', 'speedx-maintenance-mode') . '" : "' . esc_html__('Enable Password', 'speedx-maintenance-mode') . '";
                }
                togglePW.addEventListener("change", updateLabelPW);
                updateLabelPW();
            }
        });
    ';

    // Pridani inline skriptu k handle
    wp_add_inline_script('spx1mm-admin-script', $inline_script);

    // Enqueue skriptu
    wp_enqueue_script('spx1mm-admin-script');
}
add_action('admin_enqueue_scripts', 'spx1mm_enqueue_admin_scripts');

// Pridani vlastniho css a js
function spx1mm_pridat_admin_scripty()
{
    $plugin_data = get_plugin_data(__FILE__);
    $verze_pluginu = $plugin_data['Version'];
    wp_enqueue_style('speedx-maintenance-mode', plugin_dir_url(__FILE__) . 'css.css', array(), $verze_pluginu);
    wp_enqueue_script('speedx-maintenance-mode', plugin_dir_url(__FILE__) . 'js.js', array('jquery'), $verze_pluginu, true);
}
add_action('admin_enqueue_scripts', 'spx1mm_pridat_admin_scripty');

// Registrace a nacteni js pro Google reCaptcha
function spx1mm_register_scripts() {
    $reca_site_key = get_option('spx1mm_reca_site_key');

    if (!empty($reca_site_key)) {

        wp_register_script(
            'google-recaptcha',
            'https://www.google.com/recaptcha/api.js?render=' . esc_attr($reca_site_key),
            [],
            '3.0',
            false // Do hlavicky
        );

        wp_register_script(
            'spx1mm-recaptcha-init',
            '', // Inline skript soubor ne-treba
            ['google-recaptcha'],
            '1.0',
            true // Do paticky
        );
        
        wp_add_inline_script(
            'spx1mm-recaptcha-init',
            "document.addEventListener('DOMContentLoaded', function () {
                const spx1mm_submitButton = document.getElementById('spx1mm_submit');
                const spx1mm_loginForm = document.getElementById('spx1mm_login');

                if (spx1mm_submitButton && spx1mm_loginForm) {
                    spx1mm_submitButton.addEventListener('click', function(event) {
                        event.preventDefault();
                        grecaptcha.execute('" . sanitize_text_field(wp_unslash($reca_site_key)) . "', { action: 'login' }).then(function (token) {
                            document.getElementById('g-recaptcha-response').value = token;
                            spx1mm_loginForm.submit();
                        });
                    });
                }
            });"
        );

        // Enqueue skripty ihned, aby byly pripravene pro vypis
        wp_enqueue_script('google-recaptcha');
        wp_enqueue_script('spx1mm-recaptcha-init');
    }
}
add_action('init', 'spx1mm_register_scripts');

// Pridani stranky do administrace
add_action('admin_menu', function () {
    // Sdilene parametry
    $page_title = 'Speedx - ' . esc_html__('Settings', 'speedx-maintenance-mode') . ' ' . esc_html__('Maintenance Mode', 'speedx-maintenance-mode');
    $menu_title = esc_html__('Maintenance Mode', 'speedx-maintenance-mode');
    $capability = 'manage_options';
    $menu_slug  = 'spx1mm-settings';
    $callback   = 'spx1mm_settings_page';

    // Pridani pod Nastaveni
    add_options_page($page_title, $menu_title, $capability, $menu_slug, $callback);

    // Pridani pod Nastroje
    add_submenu_page('tools.php', $page_title, $menu_title, $capability, $menu_slug, $callback);
});

// Pridani odkazu na nastaveni do seznamu pluginu
add_filter('plugin_action_links_speedx-maintenance-mode/speedx-maintenance-mode.php', function ($links) {
    // Vytvoreni odkazu na nastaveni
    $settings_link = '<a href="' . admin_url('tools.php?page=spx1mm-settings') . '">Settings</a>';
    
    // Pridani odkazu na zacatek seznamu odkazu
    array_unshift($links, $settings_link);
    
    return $links;
});

// Vlastni sanitizer
// Seznamy povolenych HTML tagu pro wp_kses()
class spx1mm_sanitizer {

    // Seznam povoleny HTMl pro chybu
    public static function spx1mm_get_allowed_html_chyba() {
        return [
            'div' => ['class' => [], 'id' => [], 'style' => []],
            'p' => ['class' => [], 'id' => [], 'style' => []],
            'a' => ['href' => [], 'title' => [], 'target' => []],
            'h3' => ['class' => [], 'id' => [], 'style' => []],
            'span' => ['class' => [], 'id' => [], 'style' => []],
            'br' => [],
            'strong' => [],
        ];
    }
    
    //seznam povolenych HTML pro bezny obsah
    public static function spx1mm_get_allowed_html_obsah() {
        return [
            'meta'  => ['name' => [], 'content' => [], 'charset' => [], 'http-equiv' => []],
            'link'  => ['rel' => [], 'href' => [], 'type' => [], 'media' => [], 'sizes' => [], 'crossorigin' => []],
            'script' => ['src' => [], 'type' => [], 'async' => [], 'defer' => []],
            'style' => [],
            'div' => ['class' => [], 'id' => [], 'style' => []],
            'p' => ['class' => [], 'id' => [], 'style' => []],
            'a' => ['href' => [], 'title' => [], 'target' => []],
            'span' => ['class' => [], 'id' => [], 'style' => []],
            'h1' => ['class' => [], 'id' => [], 'style' => []],
            'h2' => ['class' => [], 'id' => [], 'style' => []],
            'h3' => ['class' => [], 'id' => [], 'style' => []],
            'h4' => ['class' => [], 'id' => [], 'style' => []],
            'h5' => ['class' => [], 'id' => [], 'style' => []],
            'h6' => ['class' => [], 'id' => [], 'style' => []],
            'strong' => ['class' => [], 'id' => [], 'style' => []],
            'em' => ['class' => [], 'id' => [], 'style' => []],
            'br' => [],
            'ul' => ['class' => [], 'id' => [], 'style' => []],
            'ol' => ['class' => [], 'id' => [], 'style' => []],
            'li' => ['class' => [], 'id' => [], 'style' => []],
            'blockquote' => [],
            'center' => [],
            'sub' => ['class' => [], 'id' => [], 'style' => []],
            'sup' => ['class' => [], 'id' => [], 'style' => []],
            'table' => ['class' => [], 'id' => [], 'style' => []],
            'tbody' => ['class' => [], 'id' => [], 'style' => []],
            'tr' => ['class' => [], 'id' => [], 'style' => []],
            'td' => ['class' => [], 'id' => [], 'style' => []],
            'thead' => ['class' => [], 'id' => [], 'style' => []],
            'hr' => ['class' => [], 'id' => [], 'style' => []],
            'del' => [],
            'b' => ['class' => [], 'id' => [], 'style' => []],
            'i' => ['class' => [], 'id' => [], 'style' => []],
            'q' => ['class' => [], 'id' => [], 'style' => []],
            's' => ['class' => [], 'id' => [], 'style' => []],
            'u' => ['class' => [], 'id' => [], 'style' => []],
            // Obrazky
            'img' => ['src' => [], 'title' => [], 'alt' => [], 'width' => [], 'height' => [], 'class' => [], 'id' => [], 'style' => [],],
            // Video
            'video'  => [ 'src' => [], 'title' => [], 'width' => [], 'height' => [], 'controls' => [], 'autoplay' => [], 'loop' => [], 'muted' => [], 'poster' => [],],
            'source' => [ 'src' => [], 'type' => [],],
            // Embed
            'embed'  => ['src' => [], 'width' => [], 'height' => [], 'allowfullscreen' => [],],
            'iframe' => ['src' => [], 'width' => [], 'height' => [], 'allowfullscreen' => [], 'frameborder' => [], 'scrolling' => [],],
            'canvas' => ['class' => [], 'id' => [], 'style' => []],
            // Formulare
            'form' => ['action' => [], 'method'  => [], 'target'  => [], 'enctype' => [], 'class' => [], 'id' => [], 'style'   => [],],
            'input'  => ['type' => [], 'id' => [], 'name' => [], 'title' => [], 'value' => [], 'checked'  => [], 'required' => [], 'disabled' => [], 'readonly' => [], 'maxlength'=> [], 'size' => [], 'class' => [], 'style' => [], 'min' => [], 'max' => [],],
            'label'  => ['for' => [], 'id' => [], 'class' => [], 'style'=> [],],
            'button' => ['type' => [], 'class' => [], 'id' => [], 'style' => [],],
            //'select' => ['name' => [], 'id' => [], 'class' => [], 'style' => [],],
            //'option' => ['value'  => [], 'selected' => [],],
            //'textarea' => ['name' => [], 'rows' => [], 'cols' => [], 'class' => [], 'style' => [],],

            // SVG
            'svg' => ['xmlns' => [], 'viewBox' => [], 'width' => [], 'height' => [], 'class' => [], 'id' => [], 'style' => [],],
            'path' => ['d' => [], 'fill' => [], 'stroke' => [], 'stroke-width' => [],],
            'circle' => ['cx' => [], 'cy' => [], 'r' => [], 'fill' => [],],
            'rect' => ['x' => [], 'y' => [], 'width' => [], 'height'=> [], 'fill'  => [],],
            'line' => ['x1' => [], 'y1' => [], 'x2' => [], 'y2' => [], 'stroke'=> [],],
            'polygon' => ['points' => [], 'fill' => [],],
            'polyline' => ['points' => [], 'stroke' => [],],
            'g' => [],
        ];
    }

    //Sanitizuje HTML obsah pomoci wp_kses() pro chybu
    public static function spx1mm_sanitize_html_chyba($html) {
        return wp_kses($html, self::spx1mm_get_allowed_html_chyba());
    }

    //Sanitizuje HTML obsah pomoci wp_kses() pro rozsireny HTML obsah
    public static function spx1mm_sanitize_html_obsah($html) {
        return wp_kses($html, self::spx1mm_get_allowed_html_obsah());
    }
}

// Vychozi HTML a obsah
class spx1mm_vychozi_obsah {
    public $spxMmDefaultTitle;
    public $spxMmDefaultContent;
    public $spxMmDefaultStyles;

    public function __construct($spxDefaultTitle, $spxDefaultContent, $spxMmDefaultStyles) {
        $this->spxMmDefaultTitle = $spxDefaultTitle;
        $this->spxMmDefaultContent = $spxDefaultContent;
        $this->spxMmDefaultStyles = $spxMmDefaultStyles;
    }
}

function spx1mm_get_vychozi_obsah() {
    $spxMmDefaultTitle = esc_html__('Maintenance Mode', 'speedx-maintenance-mode');
    $spxMmDefaultContent = '<h1>' . esc_html__('Web is currently in maintenance mode', 'speedx-maintenance-mode') . '</h1><p>' . esc_html__('We apologize for the temporary unavailability. Please try again later.', 'speedx-maintenance-mode') . '</p>';
    $spxMmDefaultStyles = 'body { font-family: Arial, sans-serif; text-align: center; padding: 50px; }
h1 { font-size: 2.5em; color: #333; }
p { font-size: 1.2em; color: #666; }
#spx1mm_login {display: flex;flex-direction: column;align-items: center;margin: 1em auto;padding: 1em;max-width: 350px;border: solid 2px #ccc;border-radius: 15px;font-size: 22px;box-shadow: 5px 5px 10px #ccc;font-weight: 600;color: #333;}
#spx1mm_login input[type=text] {margin: 1em 0;width: 100%;padding: .5em;font-size: large;text-align: center;}
#spx1mm_login button {font-size: large;padding: .5em 2em;font-weight: 600;}';

    return new spx1mm_vychozi_obsah($spxMmDefaultTitle, $spxMmDefaultContent, $spxMmDefaultStyles);
}

// Pridani informace o stavu MM do admin baru
function spx1mm_admin_bar_menu($wp_admin_bar) {
    // Načtení globálního objektu WP_Admin_Bar
    global $wp_admin_bar;

    if (!is_admin() || !current_user_can('manage_options')) {
        return;
    }

    // Získání aktuálního stavu maintenance mode
    $is_enabled = get_option('spx1mm_enabled', false);

    // Nastavení barvy a textu podle stavu
    $status_text = $is_enabled ? 'On' : 'Off';
    $status_color = $is_enabled ? '#27ae60' : '#c0392b';
    $menu_title = '<span style="color: ' . esc_html($status_color) . '; font-weight: bold;">' . esc_html__('Maintenance', 'speedx-maintenance-mode') . ': ' . esc_html($status_text) . '</span>';

    // Aktualizace existujícího uzlu v admin baru
    $wp_admin_bar->add_node([
        'id'    => 'spx1mm_status',
        'title' => spx1mm_sanitizer::spx1mm_sanitize_html_chyba($menu_title),
        'href'  => admin_url('options-general.php?page=spx1mm-settings'),
        'meta'  => ['title' => esc_attr__('Click to set the maintenance mode', 'speedx-maintenance-mode')],
    ]);
}
add_action('admin_bar_menu', 'spx1mm_admin_bar_menu', 100);

// Overeni hesla z frontendu add_action = WordPress inicializace pro zprac form mimo admin
add_action('init', function() {
    if (isset($_SERVER['REQUEST_METHOD']) && $_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['spx1mm_unlock'])) {
        // Overeni nonce
        if (isset($_POST['spx1mm_heslo_nonce']) && wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['spx1mm_heslo_nonce'])), 'spx1mm_heslo')) {
            // Overeni poctu zadani hesla
            $max_pokusy = 3;
            $lockout_time = 15 * 60;
            $salt = 'spx1mm_secret_salt';
            $today = gmdate('Y-m-d');

            // Testovaci cookie kdyz nic tak konec
            if (!isset($_COOKIE['spx1mm_test'])) {
                $GLOBALS['spx1mm_chyba'] .= '<div class="error"><p>' . esc_html__('Your browser does not support cookies. Cannot verify the entry code.', 'speedx-maintenance-mode') . '</p></div>' . "\n";
                return;
            }

            // Google reCAPTCHA
            $reca_site_key = esc_attr(get_option('spx1mm_reca_site_key'));
            $reca_secret_key = esc_attr(get_option('spx1mm_reca_secret_key'));
            $reca_score = esc_attr(trim(get_option('spx1mm_reca_score')));

            // Default pro rcaptcha score, aby to vzdy fungovalo
            if (!is_numeric($reca_score) || $reca_score < 0.0 || $reca_score > 1.0) {
                $reca_score = '0.5';
            }

            if (!empty($reca_site_key) && !empty($reca_secret_key)) {
                $recaptcha_response = isset($_POST['g-recaptcha-response']) ? sanitize_text_field(wp_unslash($_POST['g-recaptcha-response'])) : '';
                $remote_ip = isset($_SERVER['REMOTE_ADDR']) ? filter_var(wp_unslash($_SERVER['REMOTE_ADDR']), FILTER_VALIDATE_IP) : '';

                // Overeni reCAPTCHA pres Google API
                $verify_url = 'https://www.google.com/recaptcha/api/siteverify';
                $response = wp_remote_post($verify_url, [
                    'body' => [
                    'secret' => $reca_secret_key,
                    'response' => $recaptcha_response,
                    'remoteip' => $remote_ip
                    ]
                ]);
    
                $response_body = wp_remote_retrieve_body($response);
                $result = json_decode($response_body, true);
    
                // Overeni skore (0.0 - 1.0)
                if (!$result['success'] || $result['score'] < $reca_score) {
                    //error_log('reCAPTCHA Error Response: ' . print_r($result, true));
                    $GLOBALS['spx1mm_chyba'] .= '<div class="error"><p>' . esc_html__('The reCAPTCHA verification failed.', 'speedx-maintenance-mode') . '</p></div>' . "\n";
                    return;
                }
            }

            // Generovani moznych hashu pro pokusy
            $valid_attempts = [];
            for ($i = 1; $i <= $max_pokusy; $i++) {
                $valid_attempts[$i] = md5($salt . $today . $i);
            }

            // Nacteni aktualniho pokusu z cookie
            $current_attempt_hash = isset($_COOKIE['spx1mm_attempt']) ? sanitize_text_field(wp_unslash($_COOKIE['spx1mm_attempt'])) : null;
            $current_attempt = array_search($current_attempt_hash, $valid_attempts);

            if ($current_attempt === false) {
                // Pokud hodnota neni platna = prvni pokus
                $current_attempt = 1;
            } else {
                // Pokud max pokus, blok dalsich pokusu
                if ($current_attempt >= $max_pokusy) {
                    // echo 'Try again in 15 minutes.';
                    return;
                }
                $current_attempt++; // Posuneme se na dalsi pokus
            }

            // Honeypot
            $tbxPindiku = isset($_POST['TBX_pindiku']) ? (int) strlen(substr(sanitize_text_field(wp_unslash($_POST['TBX_pindiku'])), 0, 8)) : 0;

            // Overeni poctu zadani a homeypot 
            if ($tbxPindiku == null) {
                $zadavane_heslo = sanitize_text_field(wp_unslash($_POST['spx1mm_unlock']));
                $passwordEnabled = get_option('spx1mm_enabled_password', false);
                $password = get_option('spx1mm_password', '');
                $urlWeb = home_url();

                if (!empty($password) && $passwordEnabled == 1 && $zadavane_heslo === $password) {
                    $remote_ip = isset($_SERVER['REMOTE_ADDR']) ? filter_var(wp_unslash($_SERVER['REMOTE_ADDR']), FILTER_VALIDATE_IP) : '';
                    $user_agent = isset($_SERVER['HTTP_USER_AGENT']) ? sanitize_text_field(wp_unslash($_SERVER['HTTP_USER_AGENT'])) : '';

                    // Over, platnosti IP a User-Agent
                    if (empty($remote_ip) || empty($user_agent)) {
                        $user_token = md5('Pseudonahodna data pro md5' . gmdate('l'));
                    } else {
                        $user_token = md5($remote_ip . $user_agent);
                    }

                    setcookie('spx1mm_unlocked', $user_token, time() + 3600, "/", "", true, true);
                    setcookie('spx1mm_attempt', '', time() - 3600, "/", "", true, true); // Reset pokusu
                    echo '<meta http-equiv="refresh" content="2;url=' . esc_html($urlWeb) . '">';
                    $GLOBALS['spx1mm_chyba'] .= '<div class="updated">
                        <p>' . spx1mm_sanitizer::spx1mm_sanitize_html_chyba(__('Access granted.<br>The page will automatically refresh in 2 seconds.<br>Or you can refresh it by clicking on the link:', 'speedx-maintenance-mode')) . '<br>
                            <a href="' . esc_html($urlWeb) . '">' . esc_html($urlWeb) . '</a>
                        </p>
                    </div>' . "\n";
                } else {
                    // Nastavení nového pokusu do cookie
                    setcookie('spx1mm_attempt', $valid_attempts[$current_attempt], time() + $lockout_time, "/", "", true, true);
                    
                    $GLOBALS['spx1mm_chyba'] .= '<div class="error"><p>' . esc_html__('Incorrect code. Remaining attempts:', 'speedx-maintenance-mode') . esc_html(max(0, $max_pokusy - $current_attempt)) . '</p></div>' . "\n";
                }
            } else {
                $GLOBALS['spx1mm_chyba'] .= '<div class="error"><p>' . esc_html__('Entry code validation error', 'speedx-maintenance-mode') . ' (H)</p></div>' . "\n";
            }
        } else {
            $GLOBALS['spx1mm_chyba'] .= '<div class="error"><p>' . esc_html__('Entry code validation error', 'speedx-maintenance-mode') . ' (N)</p></div>' . "\n";
        }
    }
});

// Funkce pro zobrazeni rezimu udrzby 
function spx1mm_maintenance_mode() {
    // Nacteni vychoziho obsahu
    $result = spx1mm_get_vychozi_obsah();

    // Ziskani nastaveni z databaze
    $enabled = get_option('spx1mm_enabled', false);

    // Gen token
    $remote_ip = isset($_SERVER['REMOTE_ADDR']) ? filter_var(wp_unslash($_SERVER['REMOTE_ADDR']), FILTER_VALIDATE_IP) : '';
    $user_agent = isset($_SERVER['HTTP_USER_AGENT']) ? sanitize_text_field(wp_unslash($_SERVER['HTTP_USER_AGENT'])) : '';
    // Over, platnosti IP a User-Agent
    if (empty($remote_ip) || empty($user_agent)) {
        $user_token = md5('Pseudonahodna data pro md5' . gmdate('l'));
    } else {
        $user_token = md5($remote_ip . $user_agent);
    }

    // Cookie data cisticka
    $cookieData = isset($_COOKIE['spx1mm_unlocked']) ? sanitize_text_field(wp_unslash($_COOKIE['spx1mm_unlocked'])) : null;

    // Pokud je rezim udrzby zapnuty a uzivatel neni prihlasen
    if ($enabled && !is_user_logged_in() && !is_admin() && (!isset($_COOKIE['spx1mm_unlocked']) || $cookieData !== $user_token)) {
        // Ziskani nastaveni z databaze
        $title = esc_attr(get_option('spx1mm_title', $result->spxMmDefaultTitle));
        $content = get_option('spx1mm_content', $result->spxMmDefaultContent);
        $styles = get_option('spx1mm_styles', $result->spxMmDefaultStyles);
        $lang = esc_attr(get_bloginfo('language'));
        $head = get_option('spx1mm_head');
        $afterBody = get_option('spx1mm_after_body');
        $beforeHtml = get_option('spx1mm_before_html');
        $passwordEnabled = get_option('spx1mm_enabled_password', false);
        $password = get_option('spx1mm_password');
        $reca_site_key = esc_attr(get_option('spx1mm_reca_site_key'));
        $reca_secret_key = esc_attr(get_option('spx1mm_reca_secret_key'));

        // Dafautlni hodnoty. Bohuzel kdyz uz bylo jednou zadano nefungje get_option('spx1mm_title', $result->spxMmDefaultTitle)
        if (empty($title)) {
            $title = $result->spxMmDefaultTitle;
        }
        if (empty($content)) {
            $content = $result->spxMmDefaultContent;
        }
        if (empty($styles)) {
            $styles = $result->spxMmDefaultStyles;
        }

        // Ujisteni ze skripty jsou ve fronte pred vygenerovanim HTML
        do_action('wp_enqueue_scripts');
        
        // Hlavicky
        header('HTTP/1.1 503 Service Temporarily Unavailable');
        header('Retry-After: 3600');

        // Overeni poctu zadani hesla
        $max_pokusy = 3;
        $salt = 'spx1mm_secret_salt';
        $today = gmdate('Y-m-d');

        // Generovani moznych hashu pro pokusy
        $valid_attempts = [];
        for ($i = 1; $i <= $max_pokusy; $i++) {
            $valid_attempts[$i] = md5($salt . $today . $i);
        }

        // Nacteni aktualniho pokusu z cookie
        $current_attempt_hash = isset($_COOKIE['spx1mm_attempt']) ? sanitize_text_field(wp_unslash($_COOKIE['spx1mm_attempt'])) : null;
        $current_attempt = array_search($current_attempt_hash, $valid_attempts);

        // Kod pro reCaptcha
        $recaCodeFrm = '';
        if (!empty($reca_site_key) && !empty($reca_secret_key)) {
            $recaCodeFrm = '<input type="hidden" name="g-recaptcha-response" id="g-recaptcha-response">';
        }

        // Kod pro overovaci formular
        $passFormHtml = '';
        // Pokud max pokus, nezobrazuj formular
        if ($current_attempt >= $max_pokusy) {
            $passFormHtml = '<div class="error"><p>' . esc_html__('The maximum number of attempts to enter the input password has been exceeded. Try again in 15 minutes.', 'speedx-maintenance-mode') . '</p></div>';
        } else {
            // Kod pro overovaci formular
            if (!empty($passwordEnabled) && !empty($password)) {
                $passFormHtml = '<form method="post" id="spx1mm_login">
                    <label for="spx1mm_unlock">' . esc_html__('Enter access code', 'speedx-maintenance-mode') . ':</label>
                    <input type="text" name="spx1mm_unlock" required>
                    <input name="TBX_pindiku" id="TBX_pindiku" title="Počet pindíků v heslu." class="TBXp" type="number" max="200">
                    ' . $recaCodeFrm . '
                    ' . wp_nonce_field('spx1mm_heslo', 'spx1mm_heslo_nonce') . '
                    <button type="submit" id="spx1mm_submit">' . esc_html__('Submit', 'speedx-maintenance-mode') . '</button>
                </form>';
            }
        }

        //echo '<p>passwordEnabled: ' . $passwordEnabled . '</p>';
        // echo '<p>password: ' . $password . '</p>';

        // Zpracovani a zobrazeni chyb
        $LocalChyba = '';
        if (strlen(isset($GLOBALS['spx1mm_chyba'])) > 0) {
            $LocalChyba = '<div class="error_cover">
                '. spx1mm_sanitizer::spx1mm_sanitize_html_chyba($GLOBALS['spx1mm_chyba']) . '
            </div>';
        }

        // HTML obsah
        echo '<!DOCTYPE html>
        <html lang="' . esc_html($lang) . '">
        <head>
            <meta charset="UTF-8">
            <meta name="viewport" content="width=device-width, initial-scale=1.0">
            <title>' . esc_html($title) . '</title>
            ' . spx1mm_sanitizer::spx1mm_sanitize_html_obsah($head);
            if (wp_script_is('google-recaptcha', 'enqueued')) {
                echo '<script src="https://www.google.com/recaptcha/api.js?render=' . sanitize_text_field(wp_unslash($reca_site_key)) . '"></script>';
            }
        echo '<style>
            ' . wp_strip_all_tags(wp_unslash($styles)) . '
            .TBXp {width:0px;border-radius:0px;height:0px;border:solid 0px #A2A2A2;font-size:0px;margin: 0;padding: 0;}
            </style>
        </head>
        <body>
            ' . (!empty($afterBody) ? spx1mm_sanitizer::spx1mm_sanitize_html_obsah($afterBody) . "\n\t" : '') . wpautop(wp_kses_post($content)) . spx1mm_sanitizer::spx1mm_sanitize_html_obsah($LocalChyba) . spx1mm_sanitizer::spx1mm_sanitize_html_obsah($passFormHtml);
            if (wp_script_is('spx1mm-recaptcha-init', 'enqueued')) {
                echo '<script>
    document.getElementById(\'spx1mm_submit\').addEventListener(\'click\', function(event) {
        event.preventDefault();
        grecaptcha.execute(\'' . sanitize_text_field(wp_unslash($reca_site_key)) . '\', {action: \'login\'}).then(function(token) {
            document.getElementById(\'g-recaptcha-response\').value = token;
            document.getElementById(\'spx1mm_login\').submit();
        });
    });
</script>';
            }
        echo '</body>
        ' . spx1mm_sanitizer::spx1mm_sanitize_html_obsah($beforeHtml) . '
        </html>';
        exit;
    }
}
add_action('template_redirect', 'spx1mm_maintenance_mode');

// Vymazani ruznych cache pluginu
require 'spx1mm-vypni-cache.php';

// Funkce pro zakazani kesovani pokud je zapnute heslo.
function spx1mm_disable_cache_on_all_pages() {
    // Ziskani nastaveni z databaze
    $enabled = get_option('spx1mm_enabled', false);
    $passwordEnabled = get_option('spx1mm_enabled_password', false);
    $password = get_option('spx1mm_password');

    if (!empty($passwordEnabled) && !empty($password)) {
        if (!defined('DONOTCACHEPAGE')) {
            define('DONOTCACHEPAGE', true);
        }
        header("Cache-Control: no-store, no-cache, must-revalidate, max-age=0");
        header("Pragma: no-cache");
    }
}
add_action('template_redirect', 'spx1mm_disable_cache_on_all_pages');

// Stranka nastaveni
function spx1mm_settings_page() {
    // Get sanitizovana zal. z URL, kdyz nic tak zal. zakladni nastaveni
    $active_tab = isset($_GET['tab']) ? sanitize_text_field(wp_unslash($_GET['tab'])) : 'basic-settings';

    // Povolene zalozky
    $allowed_tabs = ['basic-settings', 'advanced-settings', 'password-settings'];
    // Kontrola zadani vs povolene
    if (!in_array($active_tab, $allowed_tabs, true)) {
        $active_tab = 'basic-settings'; // Vychozi pokud je zadana neplatna
    }

    // Uloz nastaveni
    if (isset($_SERVER['REQUEST_METHOD']) && $_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['edit_html_nonce'])) {
        if (!check_admin_referer('edit-html', 'edit_html_nonce')) {
            wp_die('Nonce edit verification failed.');
        }

        // Tab 1 zakladni nastaveni
        if (isset($_POST['spx1mm_tab1'])) {
            if (isset($_POST['spx1mm_enabled'])) {
                update_option('spx1mm_enabled', true);

                // Vymazani chache pri aktivaci maintenance mode
                spx1mm_vypniCache();
            } else {
                update_option('spx1mm_enabled', false);
            }

            if (isset($_POST['spx1mm_title'])) {
                update_option('spx1mm_title', esc_html(wp_unslash($_POST['spx1mm_title'])));
            }
            
            if (isset($_POST['spx1mm_content'])) {
                update_option('spx1mm_content', wp_kses_post(wp_unslash($_POST['spx1mm_content'])));
            }

            echo '<div class="updated"><p>' . esc_html__('The settings have been saved', 'speedx-maintenance-mode') . '.</p><meta http-equiv="refresh" content="1;url="></div>';
            $active_tab = 'basic-settings';
        }
        
        if (isset($_POST['spx1mm_tab2'])) {
            // Tab 2 zakladni nastaveni
            if (isset($_POST['spx1mm_styles'])) {
                $sanitized_spx1mm_styles = wp_strip_all_tags(wp_unslash($_POST['spx1mm_styles']));
                update_option('spx1mm_styles', $sanitized_spx1mm_styles);
            }

            update_option('spx1mm_head', spx1mm_sanitizer::spx1mm_sanitize_html_obsah(wp_unslash($_POST['spx1mm_head'])));
            update_option('spx1mm_after_body', spx1mm_sanitizer::spx1mm_sanitize_html_obsah(wp_unslash($_POST['spx1mm_after_body'])));
            update_option('spx1mm_before_html', spx1mm_sanitizer::spx1mm_sanitize_html_obsah(wp_unslash($_POST['spx1mm_before_html'])));

            echo '<div class="updated"><p>' . esc_html__('The settings have been saved', 'speedx-maintenance-mode') . '.</p></div>';
            $active_tab = 'advanced-settings';
        }

        if (isset($_POST['spx1mm_tab3'])) {
            // Tab 3 nastaveni hesla
            if (isset($_POST['spx1mm_enabled_password'])) {
                update_option('spx1mm_enabled_password', true);
            } else {
                update_option('spx1mm_enabled_password', false);
            }

            // sanitize_text_field(wp_unslash($_POST['spx1mm_reca_secret_key']))
            update_option('spx1mm_password', stripslashes(wp_unslash($_POST['spx1mm_password'])));
            update_option('spx1mm_reca_site_key', sanitize_text_field(wp_unslash($_POST['spx1mm_reca_site_key'])));
            update_option('spx1mm_reca_secret_key', sanitize_text_field(wp_unslash($_POST['spx1mm_reca_secret_key'])));

            $reca_score = isset($_POST['spx1mm_reca_score']) ? floatval($_POST['spx1mm_reca_score']) : 0.5;
            $reca_score = min(1.0, max(0.0, $reca_score)); // Omezi hodnotu na 0.0 - 1.0
            update_option('spx1mm_reca_score', $reca_score);

            echo '<div class="updated"><p>' . esc_html__('The settings have been saved', 'speedx-maintenance-mode') . '.</p></div>';
            $active_tab = 'password-settings';
        }
    }

    // Nacteni vychoziho obsahu
    $result = spx1mm_get_vychozi_obsah();

    // Nacteni aktualnich hodnot
    $title = get_option('spx1mm_title', $result->spxMmDefaultTitle);
    $enabled = get_option('spx1mm_enabled', false);
    $styles = get_option('spx1mm_styles', $result->spxMmDefaultStyles);
    $content = get_option('spx1mm_content', $result->spxMmDefaultContent);
    $head = get_option('spx1mm_head');
    $afterBody = get_option('spx1mm_after_body');
    $beforeHtml = get_option('spx1mm_before_html');
    $passwordEnabled = get_option('spx1mm_enabled_password', false);
    $password = get_option('spx1mm_password');
    $reca_site_key = get_option('spx1mm_reca_site_key');
    $reca_secret_key = get_option('spx1mm_reca_secret_key');
    $reca_score = get_option('spx1mm_reca_score');

    // Dafautlni hodnoty sychrovka
    if ($title <= null) {
        $title = $result->spxMmDefaultTitle;
    }
    if ($content <= null) {
        $content = $result->spxMmDefaultContent;
    }
    if ($styles <= null) {
        $styles = $result->spxMmDefaultStyles;
    }
    ?>
<div class="wrap">
    <h1><?php esc_html_e('Settings', 'speedx-maintenance-mode'); ?> <?php esc_html_e('Maintenance Mode', 'speedx-maintenance-mode'); ?></h1>
    <!-- Zalozky -->
    <div class="spx-tabs">
        <div class="spx-tab <?php echo esc_attr(($active_tab === 'basic-settings') ? sanitize_text_field('active') : ''); ?>" data-tab="basic-settings"><?php esc_html_e('Basic settings', 'speedx-maintenance-mode'); ?></div>
        <div class="spx-tab <?php echo esc_attr(($active_tab === 'advanced-settings') ? sanitize_text_field('active') : ''); ?>" data-tab="advanced-settings"><?php esc_html_e('Detailed settings', 'speedx-maintenance-mode'); ?></div>
        <div class="spx-tab <?php echo esc_attr(($active_tab === 'password-settings') ? sanitize_text_field('active') : ''); ?>" data-tab="password-settings"><?php esc_html_e('Password settings', 'speedx-maintenance-mode'); ?></div>
    </div>
    <!-- Zakladni nastaveni -->
    <div class="spx-tab-content <?php echo esc_attr(($active_tab === 'basic-settings') ? sanitize_text_field('active') : ''); ?>" id="basic-settings">
        <form method="post">
            <?php wp_nonce_field('edit-html', 'edit_html_nonce'); ?>
            <label class="spx-switch">
                <input type="checkbox" name="spx1mm_enabled" id="spx1mm_enabled" value="1" <?php checked($enabled); ?>>
                <span class="spx-slider"></span>
            </label>
            <label for="spx1mm_enabled" id="spx1mm_label"> <?php echo esc_html($enabled ? __('Disable Maintenance Mode', 'speedx-maintenance-mode') : __('Enable Maintenance Mode', 'speedx-maintenance-mode')); ?></label>
            <h3 class="spx-h3-mb0"><?php esc_html_e('Title', 'speedx-maintenance-mode'); ?></h3>
            <p class="description"><?php esc_html_e('Displays in the top bar/tab of the browser', 'speedx-maintenance-mode'); ?></p>
            <input type="text" name="spx1mm_title" id="spx1mm_title" class="tbx99" value="<?php echo esc_textarea($title); ?>">
            <h3 class="spx-h3-mb0"><?php esc_html_e('Content', 'speedx-maintenance-mode'); ?></h3>
            <p class="description"><?php esc_html_e('Enter the content that will be displayed to unlogged users', 'speedx-maintenance-mode'); ?></p>
            <?php
                // WP system wysiwyg editor
                wp_editor(
                    wp_kses_post($content), // Vychozi hodnota (obsah)
                    'spx1mm_content', // ID pole
                        ['textarea_name' => 'spx1mm_content','media_buttons' => true,'textarea_rows' => 10,'editor_class' => 'large-text','tinymce' => true,'editor_height' => 425,]
                );
            ?>
            <p class="submit">
                <input type="hidden" name="spx1mm_tab1" value="1">
                <button type="submit" class="button button-primary"><?php esc_html_e('Save Changes', 'speedx-maintenance-mode'); ?></button>
            </p>
        </form>
    </div>
    <!-- Detailni nastaveni -->
    <div class="spx-tab-content <?php echo esc_attr(($active_tab === 'advanced-settings') ? sanitize_text_field('active') : ''); ?>" id="advanced-settings">
        <form method="post">
            <?php wp_nonce_field('edit-html', 'edit_html_nonce'); ?>
            <h3 class="spx-h3-mb0"><?php esc_html_e('CSS styles', 'speedx-maintenance-mode'); ?></h3>
            <p class="description"><?php esc_html_e('Specify custom CSS styles for the maintenance page', 'speedx-maintenance-mode'); ?>.</p>
            <textarea name="spx1mm_styles" id="spx1mm_styles" rows="10" cols="50" class="large-text"><?php echo esc_textarea($styles); ?></textarea>
            <h3 class="spx-h3-mb0"><?php esc_html_e('Head codes', 'speedx-maintenance-mode'); ?></h3>
            <p class="description"><?php esc_html_e('Meta tags and other codes inserted between tags', 'speedx-maintenance-mode'); ?> &#60;head&#62;&#60;/head&#62;</p>
            <textarea name="spx1mm_head" id="spx1mm_head" rows="10" cols="50" class="large-text"><?php echo esc_textarea($head); ?></textarea>
            <h3 class="spx-h3-mb0"><?php esc_html_e('After the tag', 'speedx-maintenance-mode'); ?> body</h3>
            <p class="description"><?php esc_html_e('HTML and other codes inserted right after the tag', 'speedx-maintenance-mode'); ?> &#60;body&#62;</p>
            <textarea name="spx1mm_after_body" id="spx1mm_after_body" rows="10" cols="50" class="large-text"><?php echo esc_textarea($afterBody); ?></textarea>
            <h3 class="spx-h3-mb0"><?php esc_html_e('Before the terminating tag', 'speedx-maintenance-mode'); ?> html</h3>
            <p class="description"><?php esc_html_e('HTML and other codes inserted before the terminating tag', 'speedx-maintenance-mode'); ?> &#60;/html&#62; (<?php esc_html_e('end of the page', 'speedx-maintenance-mode'); ?>).</p>
            <textarea name="spx1mm_before_html" id="spx1mm_before_html" rows="10" cols="50" class="large-text"><?php echo esc_textarea($beforeHtml); ?></textarea>
            <p class="submit">
                <input type="hidden" name="spx1mm_tab2" value="2">    
                <button type="submit" class="button button-primary"><?php esc_html_e('Save Changes', 'speedx-maintenance-mode'); ?></button>
            </p>
        </form>
    </div>
    <!-- Nastaveni pristupoveho hesla -->
    <div class="spx-tab-content <?php echo esc_attr(($active_tab === 'password-settings') ? sanitize_text_field('active') : ''); ?>" id="password-settings">
        <form method="post">
            <?php wp_nonce_field('edit-html', 'edit_html_nonce'); ?>
            <h3 class="spx-h3-mb0"><?php esc_html_e('Setting the input password.', 'speedx-maintenance-mode'); ?></h3>
            <p class="description"><?php esc_html_e('By enabling and setting an entry password, you allow visitors to turn off maintenance mode and see the full content and graphics of the site.', 'speedx-maintenance-mode'); ?>.</p>
            <label class="spx-switch">
                <input type="checkbox" name="spx1mm_enabled_password" id="spx1mm_enabled_password" value="1" <?php checked($passwordEnabled); ?>>
                <span class="spx-slider"></span>
            </label>
            <label for="spx1mm_enabled_password" id="spx1mm_label_password"> <?php esc_html($passwordEnabled ? __('Disable Password', 'speedx-maintenance-mode') : __('Enable Password', 'speedx-maintenance-mode')); ?></label>
            <h3 class="spx-h3-mb0"><?php esc_html_e('Password', 'speedx-maintenance-mode'); ?></h3>
            <input type="text" name="spx1mm_password" id="spx1mm_password" class="tbx99" value="<?php echo esc_textarea($password); ?>">
            <h3 class="spx-h3-mb0">Google reCaptcha</h3>
            <p class="description"><?php esc_html_e('Filling in both reCaptcha keys will activate Google reCaptcha and add an extra layer of protection to the login form.', 'speedx-maintenance-mode'); ?>.</p>
            Site Key: <input type="text" name="spx1mm_reca_site_key" id="spx1mm_reca_site_key" class="tbx99" value="<?php echo esc_textarea($reca_site_key); ?>">
            Secret Key: <input type="text" name="spx1mm_reca_secret_key" id="spx1mm_reca_secret_key" class="tbx99" value="<?php echo esc_textarea($reca_secret_key); ?>">
            <p class="description"><?php esc_html_e('reCaptcha Score from 0.0 to 1.0. 0.0-0.1 = Probable bot, 0.3 = Suspicious user (anonymous VPN, bot, robotic behavior), 0.9-0.5 = Probably human, 1.0 = Definitely human.', 'speedx-maintenance-mode'); ?>.</p>
            Score: <input type="number" id="spx1mm_reca_score" name="spx1mm_reca_score" step="0.1" min="0.0" max="1.0" class="tbx50" value="<?php echo floatval(min(1.0, max(0.0, $reca_score))); ?>">
            <p class="submit">
                <input type="hidden" name="spx1mm_tab3" value="3">    
                <button type="submit" class="button button-primary"><?php esc_html_e('Save Changes', 'speedx-maintenance-mode'); ?></button>
            </p>
            </br>
            <div class="description">
                <h3>⚠️ <?php esc_html_e('Warning!', 'speedx-maintenance-mode'); ?> ⚠️</h3>
                <p><?php esc_html_e('- When the input password is turned on and filled in, the cache is globally disabled. Thus, the website will not use faster loading of content from the cache. This is a necessary precaution against displaying content from the cache instead of verifying the input with a password.', 'speedx-maintenance-mode'); ?></p>
                <p><?php echo spx1mm_sanitizer::spx1mm_sanitize_html_chyba(__('- While this plugin includes basic security measures, it is not a sophisticated method for preventing unauthorized access. This system is designed only for simple access restriction to a staging or development version of your website. It does not replace WordPress user authentication.<br>If you require a higher level of security, we recommend using advanced methods such as WordPress user authentication or additional security plugins.', 'speedx-maintenance-mode')); ?></p>
            </div>
        </form>
    </div>
</div>
<?php
}